import { Entity, EntityDamageCause, ItemDurabilityComponent, ItemStack, system, TicksPerSecond, world } from "@minecraft/server";
import { Helper } from "../utils/Helper";
import { PlayerUtils } from "../utils/PlayerUtils";
const ITEM_ID = "sf_nba:tooth_dagger";
const ENTITY_ID = "sf_nba:tooth_dagger";
const FUTURE_DAMAGE_PROP = "sf_nba:future_damage";
const FUTURE_REMOVE_DYN_PROP = "sf_nba:to_remove";
const ON_TOOTH_DAGGER_HIT_GROUND_EVENT = "sf_nba:on_tooth_dagger_hit_ground_event";
const DAGGER_DAMAGE = 4;
const PROJECTILE_STRENGTH = 2;
export class ToothDagger {
  static blockInteractions = new Set();
  static usedDaggerTick = {};
  constructor() {
    this.onUse = this.onUse.bind(this);
    system.run(() => {
      this.#initEvents();
    });
  }
  #initEvents() {
    this.#onHitGround();
    world.afterEvents.projectileHitEntity.subscribe(this.#handleEntityHit.bind(this));
  }
  onUse({ source: player }) {
    const { dimension, location: pos } = player;
    if (ToothDagger.usedDaggerTick[player.id] === system.currentTick) return;
    if (ToothDagger.blockInteractions.has(player.id)) {
      ToothDagger.blockInteractions.delete(player.id);
      return;
    }
    ToothDagger.usedDaggerTick[player.id] = system.currentTick;
    const damageToBeApplied = PlayerUtils.damageMainhandItemStack(player, true, 1, false) ?? 0;
    Helper.decreaseMainhandItemStack(player);
    dimension.playSound("mob.snowgolem.shoot", pos, { pitch: 0.8 });
    this.#spawnAndThrowDagger(player, damageToBeApplied);
  }
  #spawnAndThrowDagger(player, damageToBeApplied) {
    const headPos = player.getHeadLocation();
    const rotation = player.getViewDirection();
    const projectile = player.dimension.spawnEntity(ENTITY_ID, {
      x: headPos.x - rotation.z * 0.3 + rotation.x * 0.2,
      y: headPos.y - 0.2,
      z: headPos.z + rotation.x * 0.3 + rotation.z * 0.2,
    });
    projectile.setRotation(player.getRotation());
    projectile.clearVelocity();
    projectile.applyImpulse({
      x: rotation.x * PROJECTILE_STRENGTH + player.getVelocity().x,
      y: rotation.y,
      z: rotation.z * PROJECTILE_STRENGTH + player.getVelocity().z
    });
    projectile.setProperty(FUTURE_DAMAGE_PROP, damageToBeApplied);
    projectile.setDynamicProperty("sf_nba:thrower", player.id);
    return projectile;
  }
  #onHitGround() {
    system.afterEvents.scriptEventReceive.subscribe(({ id, sourceEntity: projectile }) => {
      if (id !== ON_TOOTH_DAGGER_HIT_GROUND_EVENT) return;
      this.#spawnDamagedThrownDagger(projectile);
    });
  }
  #handleEntityHit(eventData) {
    const entity = eventData.projectile;
    const hitEntityResult = eventData.getEntityHit();
    if (!hitEntityResult) return;
    const target = hitEntityResult.entity;
    if (!entity || !target) return;
    const identifier = entity.typeId;
    if (identifier !== ENTITY_ID) return;
    system.runTimeout(() => {
      target.applyDamage(DAGGER_DAMAGE, {
        cause: EntityDamageCause.entityAttack, 
      });
      this.#spawnDamagedThrownDagger(entity);
    }, 0.11 * TicksPerSecond);
  }
  #spawnDamagedThrownDagger(projectile, remove = true) {
    if (!projectile) return;
    if (projectile.getDynamicProperty(FUTURE_REMOVE_DYN_PROP)) return;
    projectile.setDynamicProperty(FUTURE_REMOVE_DYN_PROP, true);
    system.runTimeout(() => {
      let futureDamage = 0;
      try {
        futureDamage = projectile.getProperty(FUTURE_DAMAGE_PROP) ?? 0;
      } catch (error) {
        futureDamage = 0; 
      }
      try {
        const { dimension, location } = projectile;
        dimension.playSound("item.trident.hit", location, { pitch: 1.5 });
        const stack = new ItemStack(ITEM_ID);
        const durability = stack.getComponent(ItemDurabilityComponent.componentId);
        if (!durability) {
          if (remove) projectile.remove();
          return;
        }
        if (futureDamage >= durability.maxDurability) {
          dimension.playSound("random.break", location);
          dimension.spawnParticle("sf_nba:tooth_dagger_break", location);
          if (remove) projectile.remove();
          return;
        }
        durability.damage = futureDamage;
        dimension.spawnItem(stack, location);
        if (remove) projectile.remove();
      } catch (error) {}
    }, 5);
  }
}